  //@author: dottore
  //@description: combines HighRes Source and Blurred version using CoC (circle of confusion); apply leaking reduction.
  //@tags: compositing, CoC
  //@credits: ShaderX3 book (chapter 4.4 ; Improved Depth-of-Field Rendering; Thorsten Scheuermann and Natalya Tararchuk)

// -------------------------------------------------------------------------------------------------------------------------------------
// PARAMETERS:
// -------------------------------------------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (DX9)
float4x4 tP: PROJECTION;   //projection matrix as set via Renderer (DX9)
float4x4 tWVP: WORLDVIEWPROJECTION;


//full resolution image
texture colorHighT <string uiname="High Res Image";>;
sampler2D tSource = sampler_state
{
    Texture = <colorHighT>;
    AddressU = Clamp;
    AddressV = Clamp;
    MinFilter = none;
    MagFilter = none;
    MipFilter = none;
};

//downsampled and pre-blurred image
texture colorLowT <string uiname="Low Res Image";>;
sampler tSourceLow = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (colorLowT);          //apply a texture to the sampler
    MipFilter = LINEAR;         //set the sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


// -------------------------------------------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// -------------------------------------------------------------------------------------------------------------------------------------

struct DOF
{
    float4 Pos  : POSITION;
    float2 TexCd : TEXCOORD;
};

DOF VSDOF(
    float4 Pos  : POSITION,
    float2 TexCd : TEXCOORD)
{
    //inititalize all fields of output struct with 0
    DOF Out = (DOF)0;

    Out.Pos  = mul(Pos, tWVP);
    Out.TexCd  = TexCd;

    return Out;
}

// -------------------------------------------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// -------------------------------------------------------------------------------------------------------------------------------------

       
// pixel size (1/image resolution) of full resolution image
float2 pixelSizeHigh;

//pixel size of the low resolution image
float2 pixelSizeLow;

//maximum CoC radius and diameter in pixels
float2 vMaxCoC = float2(5.0, 10.0);

//scale factor for maximum CoC size on low res. image
float radiusScale = 0.4;

//float4 PoissonDOFFilter (float2 texCoord)
float4 PSDOF(DOF In): COLOR
{

//contains poisson-distributed positions on the unit circle
float2 poisson[8] = {
       float2(  0.0     ,       0.0     ),
       float2(  0.527837,      -0.085868),
       float2( -0.040088,       0.536087),
       float2( -0.670445,      -0.179949),
       float2( -0.419418,      -0.616039),
       float2(  0.440453,      -0.639399),
       float2( -0.757088,       0.349334),
       float2(  0.574619,       0.685879),
       };
       
 float4 cOut;
 float discRadius;
 float discRadiusLow;
 float centerDepth;
 
 cOut = tex2D(tSource, In.TexCd);
 centerDepth = cOut.a;        //save its depth
 
 //convert depth into blur radius in pixels
 discRadius = abs(cOut.a * vMaxCoC.y - vMaxCoC.x);
 
 //compute disc radius on low-res mage
 discRadiusLow = discRadius * radiusScale;
 
 //reuse cOut as an accumulator
 cOut = 0;
 
 for(int t = 0; t < 8; t++)       // ..; t<x ;..  x= NUM_TAPS
 {
  //fetch low-res tap
  float2 coordLow = In.TexCd + (pixelSizeLow * poisson[t]
                  * discRadiusLow);
  float4 tapLow = tex2D(tSourceLow, coordLow);
  
  //fetch high-res tap
  float2 coordHigh = In.TexCd + (pixelSizeHigh * poisson[t]
                   * discRadiusLow);
  float4 tapHigh = tex2D(tSource, coordHigh);
  
  //put tap blurriness into [0, 1] range
  float tapBlur = abs(tapHigh.a * 2.0 - 1.0);
  
  //mix low- and hi-res taps based on tap blurriness
  float4 tap = lerp(tapHigh, tapLow, tapBlur);
  
  //apply leaking reduction: lower weight for taps that are
  //closer than the center tap and in focus
  tap.a = (tap.a >= centerDepth) ? 1.0 : abs(tap.a * 2.0 - 1.0);
  
  //accumulate
  cOut.rgb +=tap.rgb * tap.a;
  cOut.a += tap.a;
 }

 //normalize and return result
 return (cOut / cOut.a);
}

 
 
 



// -------------------------------------------------------------------------------------------------------------------------------------
// TECHNIQUES:
// -------------------------------------------------------------------------------------------------------------------------------------

technique TSimpleShader
{
    pass P0
    {
        VertexShader = compile vs_1_0 VSDOF();
        PixelShader  = compile ps_3_0 PSDOF();
    }
}
